<?php

namespace App\Http\Controllers;

use App\Models\Blog;
use App\Models\Category;
use App\Services\VimeoService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Validator;
use Log;
use RealRashid\SweetAlert\Facades\Alert;
use Str;

class BlogController extends Controller
{

    protected $vimeo;

    public function __construct(Vimeoservice $vimeoservice) {
        $this->vimeo = $vimeoservice;
    }

    public function index(Request $request)
    {

        $page = $request->page;

        if(isset($page) && is_numeric($page) && $page>0) {

            $records_to_show = 25;
            $records_to_skip = ($page - 1) * $records_to_show;

            $query = Blog::query();

            $all_records_count = $query->count();

            $num_of_pages = (int) ceil($all_records_count / $records_to_show);
            
            $blogs = $query->skip($records_to_skip)
            ->take($records_to_show)
            ->with('category')
            ->get();

            return view('blogs.index', compact('blogs','num_of_pages', 'page'));

            
        } else {
            return redirect()->route('blogs.index', ['page'=>1]);
        }

    }

    public function create() {

        if(session()->has('success')) {
            Alert::success('success', session()->get('success'));
        }

        if(session()->has('error')) {
            Alert::error('error', session()->get('error'));
        }

        $categories = Category::where('deleted', 0)->select('id', 'name')->get();

        return view('blogs.form', compact('categories'));
    }

    public function store(Request $request) {
        $insert = 0;
        $error = '';

        try {

            $this->requestValidations();

            $inputs = $request->all();

            // Upload Main Image
            $imgName = $this->uploadFile($request, 'main_image', 'blogs_imgs', 'required|mimes:jpeg,jpg,png,webp,avif|max:1024');
            $inputs['main_image'] = $imgName;

            // Store
            $insert = Blog::create($inputs);

        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors())->withInput();

        } catch (\Exception $e) {
            $error = $e->getMessage();
        }

        if($insert) {
            Session()->flash('success' , 'Record Created Successfully !');
        } else {
            Log::error($error);
            Session()->flash('error' , 'Something went wrong.');
        }
       
        return redirect()->back();
    }

    public function show($id) {
        
        if(session()->has('success')) {
            Alert::success('success', session()->get('success'));
        }

        if(session()->has('error')) {
            Alert::error('error', session()->get('error'));
        }

        $blog = Blog::findOrFail($id);
        $categories = Category::where('deleted', 0)->select('id', 'name')->get();

        return view('blogs.form', compact('blog', 'categories'));
    }

    public function update(Request $request, Blog $blog, $id)
    {

        $status = 0;
        $error = '';

        try {

            $this->requestValidations($id);
            $blog = Blog::findOrFail($id);

            $inputs = $request->all();

            if($request->main_image != null) {
                // Upload Image
                $imgName = $this->uploadFile($request, 'main_image', 'blogs_imgs', 'required|mimes:jpeg,jpg,png,webp,avif|max:1024');
                $inputs['main_image'] = $imgName;
            }

            $blog->fill($inputs);

            if($blog->save()) {
                $status = 1;
            }
            
        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors())->withInput();

        } catch (\Exception $e) {
            $error = $e->getMessage();
        }

        if($status) {
            Session()->flash('success' , 'Record Updated Successfully !');
        } else {
            Log::error($error);
            Session()->flash('error' , 'Something went wrong.');
        }
       
        return redirect()->back();
        
    }

    public function destroy(Request $request)
    {
        $status = 0;
        $message = 'Something went wrong !';

        $delete = Blog::where('id', $request->record_id)->delete();

        if($delete) {

            $status = 1;
            $message = "Record Deleted Successfully !";
        }

        return response()->json([
            'status'=>$status,
            'message'=>$message
        ]);
    }

    public function requestValidations($record_id = null) {

        $slugUniqueRule = Rule::unique('blogs', 'slug')->ignore($record_id);

        return request()->validate([
            'title' => ['required', 'string', 'max:256'],
            'category_id' => ['required', 'numeric'],
            'overview' => ['required'],
            'description' => ['required'],
        ]);
    }

    public function uploadFile(Request $request, $file_name, $folder_name, $rules) {

        // Get the uploaded file
        $file = $request->file($file_name);
    
        // Validation rules for general files
        $fileRules = [
            $file_name => $rules,
        ];
    
        // Validate the file input
        $this->validate($request, $fileRules);
    
        // Generate a unique name for the file
        $fileName = uniqid().'_'.Str::random(6).'.'.$file->getClientOriginalExtension();
    
        // Specify the directory where you want to save the file
        $directory = 'uploads/'.$folder_name;
    
        // Create the directory if it doesn't exist
        File::makeDirectory($directory, $mode = 0777, true, true);
    
        // Move the uploaded file to the specified directory
        $file->move($directory, $fileName);
    
        // Return the path of the uploaded file
        return $folder_name.'/'.$fileName;
    }

    public function uploadVimeoVideo(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'video' => 'required|mimes:mp4,mov,avi|max:50000',
            'title' => 'required',
            'description' => 'nullable'
        ]);

        if(!$validator->fails()) {
            $filePath = $request->file('video')->getPathname();
            $title = $request->title;
            $description = $request->description;

            $blog = Blog::find($request->record_id);

            if($blog) {
                $response = $this->vimeo->uploadVideo($filePath, $title, $description);
                $videoId = str_replace("/videos/", "", $response);

                if($videoId) {

                    $getVideo =  $this->vimeo->getVideo($videoId);

                    $player_embed_url = isset($getVideo['body']['player_embed_url']) ? $getVideo['body']['player_embed_url'] : '';

                    $blog->fill(["video"=>$videoId, "player_embed_url" => $player_embed_url]);
                    if($blog->save()) {
                        session()->flash('success' , 'Video Uplaoded Successfully.');
                    } else {
                        session()->flash('error' , 'Failed to upload video.');
                    }
                } else {
                    session()->flash('error' , 'Failed to upload to vimue.');
                }
            } else {
                session()->flash('error' , 'Record Not Found.');
            }

        } else {
            session()->flash('error' , $validator->errors()->first());
        }
 
        return redirect()->back();
    }
    

}
