<?php

namespace App\Http\Controllers;

use App\Models\Role;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use RealRashid\SweetAlert\Facades\Alert;

class UserController extends Controller
{
    
    public function index()
    {
        if (!$this->privilegeChecker('viewUsers')) {
            return redirect()->route('admin');
        }

        $users = User::all();

        if(session()->has('success')) {
            Alert::success('Success', session()->get('success'));
        }

        return view('users.index')->with('users', $users);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        if (!$this->privilegeChecker('addUsers')) {
            return redirect()->route('admin');
        }

        if(session()->has('success')) {
            Alert::success('success', session()->get('success'));
        }

        $roles = Role::all();

        return view('users.form', compact('roles'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if (!$this->privilegeChecker('addUsers')) {
            return redirect()->route('admin');
        }

        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email'],
            'password' => ['required', 'string', 'min:8'],
            'role_id' => ['required'],
        ]);
        
        User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role_id' => $request->role_id,
        ]);


        Session()->flash('success' , 'User has been created successfully !');

        return redirect()->back();
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        if (!$this->privilegeChecker('viewUsers')) {
            return redirect()->route('admin');
        }

        if(session()->has('success')) {
            Alert::success('success', session()->get('success'));
        }

        $user = User::findOrFail($id);

        $roles = Role::all();

        return view('users.form', compact('user', 'roles'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {

        if (!$this->privilegeChecker('updateUsers')) {
            return redirect()->route('admin');
        }
        
        $user = User::findOrFail($id);

        if($request->password != null) {

            $request->validate([
                'name' => ['required', 'string', 'max:255'],
                'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,'.$user->id.',id'],
                'password' => ['required', 'string', 'min:8'],
                'role_id' => ['required'],
            ]);

            $password = Hash::make($request->password);

        } else {
            $request->validate([
                'name' => ['required', 'string', 'max:255'],
                'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,'.$user->id.',id'],
                'role_id' => ['required'],
            ]);

            $password = $user->password;
        }
        
        $user->fill([
            'name' => $request->name,
            'password' => $password,
            'email' => $request->email,
            'role_id' => $request->role_id,
        ])->save();

        Session()->flash('success' , 'User has been updated successfully !');

        return redirect()->back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        if (!$this->privilegeChecker('deleteUsers')) {
            return redirect()->route('admin');
        }

        User::where('id', $id)->delete();
        
        return response()->json(['status'=> 'User has been deleted successfully !']);
    }

    public function privilegeChecker($privilege) {

        $allow = 0;

        $user_privileges_arr = app('shared')->get('user_privileges_arr');
        $authUser = app('shared')->get('user');

        if ($authUser->role->class == 'A' || in_array($privilege, $user_privileges_arr) ) {
            $allow = 1;
        }

        return $allow;
    }
}
